const express = require('express');
const fs = require('fs');
const path = require('path');
const cors = require('cors');
const multer = require('multer');

const app = express();
const PORT = process.env.PORT || 3000;

// Middleware
app.use(cors());
app.use(express.static('public'));
app.use(express.json());

// Create logs directory if it doesn't exist
const logsDir = path.join(__dirname, 'logs');
if (!fs.existsSync(logsDir)) {
  fs.mkdirSync(logsDir, { recursive: true });
}

// Enhanced logging
const log = (message, level = 'INFO') => {
  const timestamp = new Date().toISOString();
  const logMessage = `[${timestamp}] ${level}: ${message}`;
  console.log(logMessage);
  
  // Also write to log file
  try {
    fs.appendFileSync(path.join(logsDir, 'app.log'), logMessage + '\n');
  } catch (err) {
    console.error('Failed to write to log file:', err);
  }
};

// API endpoint to list music files
app.get('/api/music-files', (req, res) => {
  try {
    const musicDir = path.join(__dirname, 'public', 'music');
    
    // Check if directory exists
    if (!fs.existsSync(musicDir)) {
      log('Music directory does not exist, creating it...', 'WARN');
      fs.mkdirSync(musicDir, { recursive: true });
      return res.json([]);
    }
    
    const files = fs.readdirSync(musicDir);
    const allowedExtensions = ['.mod', '.xm', '.it', '.s3m', '.mid', '.midi', '.sf2'];
    
    const musicFiles = files
      .filter(file => {
        const ext = path.extname(file).toLowerCase();
        return allowedExtensions.includes(ext);
      })
      .map(file => {
        const filePath = path.join(musicDir, file);
        const stats = fs.statSync(filePath);
        const ext = path.extname(file).toLowerCase();
        
        return {
          filename: file,
          size: stats.size,
          modified: stats.mtime,
          type: ['.mod', '.xm', '.it', '.s3m'].includes(ext) ? 'tracker' : 
                ext === '.sf2' ? 'soundfont' : 'midi',
          displaySize: formatFileSize(stats.size)
        };
      })
      .sort((a, b) => a.filename.localeCompare(b.filename));
    
    log(`Found ${musicFiles.length} music files`);
    res.json(musicFiles);
    
  } catch (error) {
    log(`Error scanning music directory: ${error.message}`, 'ERROR');
    res.status(500).json({ error: 'Unable to scan music directory' });
  }
});

// API endpoint to get file info
app.get('/api/file-info/:filename', (req, res) => {
  try {
    const filename = req.params.filename;
    const filePath = path.join(__dirname, 'public', 'music', filename);
    
    if (!fs.existsSync(filePath)) {
      return res.status(404).json({ error: 'File not found' });
    }
    
    const stats = fs.statSync(filePath);
    const ext = path.extname(filename).toLowerCase();
    
    res.json({
      filename,
      size: stats.size,
      displaySize: formatFileSize(stats.size),
      modified: stats.mtime,
      type: ['.mod', '.xm', '.it', '.s3m'].includes(ext) ? 'tracker' : 
            ext === '.sf2' ? 'soundfont' : 'midi'
    });
    
  } catch (error) {
    log(`Error getting file info: ${error.message}`, 'ERROR');
    res.status(500).json({ error: 'Unable to get file info' });
  }
});

// API endpoint to list soundfonts
app.get('/api/soundfonts', (req, res) => {
  try {
    const soundfontsDir = path.join(__dirname, 'public', 'soundfonts');
    
    if (!fs.existsSync(soundfontsDir)) {
      return res.json([]);
    }
    
    const files = fs.readdirSync(soundfontsDir);
    const soundfonts = files
      .filter(file => path.extname(file).toLowerCase() === '.sf2')
      .map(file => {
        const filePath = path.join(soundfontsDir, file);
        const stats = fs.statSync(filePath);
        
        return {
          filename: file,
          size: stats.size,
          displaySize: formatFileSize(stats.size),
          modified: stats.mtime
        };
      });
    
    res.json(soundfonts);
    
  } catch (error) {
    log(`Error listing soundfonts: ${error.message}`, 'ERROR');
    res.status(500).json({ error: 'Unable to list soundfonts' });
  }
});

// Upload configuration
const storage = multer.diskStorage({
  destination: (req, file, cb) => {
    const ext = path.extname(file.originalname).toLowerCase();
    
    // Determine destination based on file type
    let destDir;
    if (ext === '.sf2') {
      destDir = path.join(__dirname, 'public', 'soundfonts');
    } else {
      destDir = path.join(__dirname, 'public', 'music');
    }
    
    if (!fs.existsSync(destDir)) {
      fs.mkdirSync(destDir, { recursive: true });
    }
    cb(null, destDir);
  },
  filename: (req, file, cb) => {
    // Sanitize filename
    const sanitized = file.originalname.replace(/[^a-zA-Z0-9.-]/g, '_');
    cb(null, sanitized);
  }
});

const upload = multer({ 
  storage,
  fileFilter: (req, file, cb) => {
    const allowedExtensions = ['.mod', '.xm', '.it', '.s3m', '.mid', '.midi', '.sf2'];
    const ext = path.extname(file.originalname).toLowerCase();
    cb(null, allowedExtensions.includes(ext));
  },
  limits: { fileSize: 200 * 1024 * 1024 } // 200MB limit for large soundfonts
});

app.post('/api/upload', upload.single('musicFile'), (req, res) => {
  if (!req.file) {
    return res.status(400).json({ error: 'No file uploaded or invalid format' });
  }
  
  const fileType = path.extname(req.file.filename).toLowerCase() === '.sf2' ? 'soundfont' : 'music';
  log(`File uploaded: ${req.file.filename} (${formatFileSize(req.file.size)}) - Type: ${fileType}`);
  
  res.json({ 
    message: 'File uploaded successfully', 
    filename: req.file.filename,
    size: req.file.size,
    displaySize: formatFileSize(req.file.size),
    type: fileType
  });
});

// Delete file endpoint
app.delete('/api/delete/:filename', (req, res) => {
  try {
    const filename = req.params.filename;
    const ext = path.extname(filename).toLowerCase();
    
    // Determine file location based on extension
    let filePath;
    if (ext === '.sf2') {
      filePath = path.join(__dirname, 'public', 'soundfonts', filename);
    } else {
      filePath = path.join(__dirname, 'public', 'music', filename);
    }
    
    if (!fs.existsSync(filePath)) {
      return res.status(404).json({ error: 'File not found' });
    }
    
    fs.unlinkSync(filePath);
    log(`File deleted: ${filename}`);
    res.json({ message: 'File deleted successfully' });
    
  } catch (error) {
    log(`Error deleting file: ${error.message}`, 'ERROR');
    res.status(500).json({ error: 'Unable to delete file' });
  }
});

// SoundFont management endpoint
app.post('/api/set-default-soundfont/:filename', (req, res) => {
  try {
    const filename = req.params.filename;
    const soundfontPath = path.join(__dirname, 'public', 'soundfonts', filename);
    
    if (!fs.existsSync(soundfontPath)) {
      return res.status(404).json({ error: 'SoundFont not found' });
    }
    
    // Create symlink or copy to default.sf2
    const defaultPath = path.join(__dirname, 'public', 'soundfonts', 'default.sf2');
    
    // Remove existing default if it exists
    if (fs.existsSync(defaultPath)) {
      fs.unlinkSync(defaultPath);
    }
    
    // Copy the selected soundfont to default
    fs.copyFileSync(soundfontPath, defaultPath);
    
    log(`Default SoundFont set to: ${filename}`);
    res.json({ message: `Default SoundFont set to ${filename}` });
    
  } catch (error) {
    log(`Error setting default SoundFont: ${error.message}`, 'ERROR');
    res.status(500).json({ error: 'Unable to set default SoundFont' });
  }
});

// Serve main page
app.get('/', (req, res) => {
  res.sendFile(path.join(__dirname, 'public', 'index.html'));
});

// Health check endpoint
app.get('/health', (req, res) => {
  const musicDir = path.join(__dirname, 'public', 'music');
  const soundfontsDir = path.join(__dirname, 'public', 'soundfonts');
  
  const musicDirExists = fs.existsSync(musicDir);
  const soundfontsDirExists = fs.existsSync(soundfontsDir);
  
  const musicFileCount = musicDirExists ? fs.readdirSync(musicDir).length : 0;
  const soundfontCount = soundfontsDirExists ? 
    fs.readdirSync(soundfontsDir).filter(f => f.endsWith('.sf2')).length : 0;
  
  res.json({ 
    status: 'healthy', 
    timestamp: new Date().toISOString(),
    uptime: process.uptime(),
    musicDirectory: musicDirExists,
    soundfontsDirectory: soundfontsDirExists,
    musicFileCount: musicFileCount,
    soundfontCount: soundfontCount,
    version: '2.0.0',
    engines: {
      openmpt: 'enabled',
      fluidsynth: soundfontCount > 0 ? 'enabled' : 'no_soundfonts'
    }
  });
});

// System info endpoint
app.get('/api/system-info', (req, res) => {
  const musicDir = path.join(__dirname, 'public', 'music');
  const soundfontsDir = path.join(__dirname, 'public', 'soundfonts');
  
  const musicStats = fs.existsSync(musicDir) ? fs.readdirSync(musicDir) : [];
  const soundfontStats = fs.existsSync(soundfontsDir) ? 
    fs.readdirSync(soundfontsDir).filter(f => f.endsWith('.sf2')) : [];
  
  res.json({
    uptime: Math.floor(process.uptime()),
    memory: process.memoryUsage(),
    platform: process.platform,
    nodeVersion: process.version,
    totalMusicFiles: musicStats.length,
    totalSoundfonts: soundfontStats.length,
    musicDiskUsage: getDiskUsage(musicDir),
    soundfontsDiskUsage: getDiskUsage(soundfontsDir),
    audioEngines: {
      openmpt: {
        status: 'available',
        formats: ['.mod', '.xm', '.it', '.s3m']
      },
      fluidsynth: {
        status: soundfontStats.length > 0 ? 'available' : 'no_soundfonts',
        formats: ['.mid', '.midi'],
        soundfonts: soundfontStats.length
      }
    }
  });
});

// Audio library info endpoint
app.get('/api/audio-libraries', (req, res) => {
  const jsDir = path.join(__dirname, 'public', 'js');
  
  const libraries = {};
  
  // Check OpenMPT
  const openmptJs = path.join(jsDir, 'libopenmpt.js');
  const openmptWasm = path.join(jsDir, 'libopenmpt.wasm');
  libraries.openmpt = {
    available: fs.existsSync(openmptJs) && fs.existsSync(openmptWasm),
    files: {
      js: fs.existsSync(openmptJs) ? formatFileSize(fs.statSync(openmptJs).size) : 'missing',
      wasm: fs.existsSync(openmptWasm) ? formatFileSize(fs.statSync(openmptWasm).size) : 'missing'
    }
  };
  
  // Check FluidSynth
  const fluidsynthJs = path.join(jsDir, 'fluidsynth.js');
  const fluidsynthWasm = path.join(jsDir, 'fluidsynth.wasm');
  libraries.fluidsynth = {
    available: fs.existsSync(fluidsynthJs) && fs.existsSync(fluidsynthWasm),
    files: {
      js: fs.existsSync(fluidsynthJs) ? formatFileSize(fs.statSync(fluidsynthJs).size) : 'missing',
      wasm: fs.existsSync(fluidsynthWasm) ? formatFileSize(fs.statSync(fluidsynthWasm).size) : 'missing'
    }
  };
  
  // Check AudioWorklet processor
  const audioWorkletProcessor = path.join(jsDir, 'audio-worklet-processor.js');
  libraries.audioWorklet = {
    available: fs.existsSync(audioWorkletProcessor),
    size: fs.existsSync(audioWorkletProcessor) ? 
      formatFileSize(fs.statSync(audioWorkletProcessor).size) : 'missing'
  };
  
  res.json(libraries);
});

// Error handling middleware
app.use((err, req, res, next) => {
  log(`Server error: ${err.message}`, 'ERROR');
  res.status(500).json({ error: 'Internal server error' });
});

// 404 handler
app.use((req, res) => {
  log(`404 - ${req.method} ${req.url}`, 'WARN');
  res.status(404).json({ error: 'Not found' });
});

// Utility functions
function formatFileSize(bytes) {
  const sizes = ['Bytes', 'KB', 'MB', 'GB'];
  if (bytes === 0) return '0 Bytes';
  const i = Math.floor(Math.log(bytes) / Math.log(1024));
  return Math.round(bytes / Math.pow(1024, i) * 100) / 100 + ' ' + sizes[i];
}

function getDiskUsage(dir) {
  try {
    if (!fs.existsSync(dir)) return { totalSize: 0, fileCount: 0 };
    
    const files = fs.readdirSync(dir);
    let totalSize = 0;
    
    files.forEach(file => {
      const filePath = path.join(dir, file);
      const stats = fs.statSync(filePath);
      totalSize += stats.size;
    });
    
    return {
      totalSize: totalSize,
      displaySize: formatFileSize(totalSize),
      fileCount: files.length
    };
  } catch (error) {
    return { totalSize: 0, fileCount: 0, error: error.message };
  }
}

// Start server
app.listen(PORT, '0.0.0.0', () => {
  log(`🎵 Fusion Music Player Server v2.0 starting on port ${PORT}`);
  log(`🌐 Access at: http://localhost:${PORT}`);
  log(`🐳 Running in Docker: ${process.env.NODE_ENV === 'production' ? 'YES' : 'NO'}`);
  
  // Create directories if they don't exist
  const musicDir = path.join(__dirname, 'public', 'music');
  const soundfontsDir = path.join(__dirname, 'public', 'soundfonts');
  
  if (!fs.existsSync(musicDir)) {
    fs.mkdirSync(musicDir, { recursive: true });
    log(`📁 Created music directory: ${musicDir}`);
  }
  
  if (!fs.existsSync(soundfontsDir)) {
    fs.mkdirSync(soundfontsDir, { recursive: true });
    log(`📁 Created soundfonts directory: ${soundfontsDir}`);
  }
  
  // Log initial file counts
  const musicFiles = fs.readdirSync(musicDir);
  const soundfontFiles = fs.readdirSync(soundfontsDir).filter(f => f.endsWith('.sf2'));
  
  log(`📀 Initial scan found ${musicFiles.length} files in music directory`);
  log(`🎼 Initial scan found ${soundfontFiles.length} SoundFont files`);
  
  // Check for audio libraries
  const jsDir = path.join(__dirname, 'public', 'js');
  const hasOpenMPT = fs.existsSync(path.join(jsDir, 'libopenmpt.js'));
  const hasFluidSynth = fs.existsSync(path.join(jsDir, 'fluidsynth.js'));
  const hasAudioWorklet = fs.existsSync(path.join(jsDir, 'audio-worklet-processor.js'));
  
  log(`🎹 Audio Libraries: OpenMPT:${hasOpenMPT?'✓':'✗'} FluidSynth:${hasFluidSynth?'✓':'✗'} AudioWorklet:${hasAudioWorklet?'✓':'✗'}`);
});

// Graceful shutdown
const gracefulShutdown = (signal) => {
  log(`🛑 Received ${signal}, shutting down gracefully`);
  process.exit(0);
};

process.on('SIGTERM', () => gracefulShutdown('SIGTERM'));
process.on('SIGINT', () => gracefulShutdown('SIGINT'));

// Handle uncaught exceptions
process.on('uncaughtException', (err) => {
  log(`Uncaught Exception: ${err.message}`, 'ERROR');
  console.error(err.stack);
  process.exit(1);
});

process.on('unhandledRejection', (reason, promise) => {
  log(`Unhandled Rejection at: ${promise} reason: ${reason}`, 'ERROR');
});
