// Fallback AudioEngine - Works without AudioWorklet
class FallbackAudioEngine {
    constructor() {
        this.audioContext = null;
        this.gainNode = null;
        this.volume = 0.5;
        this.isPlaying = false;
        this.isPaused = false;
        this.currentTime = 0;
        this.duration = 0;
        
        // Audio engines
        this.modulePlayer = null;
        this.chiptunePlayer = null;
        this.synthesizer = null;
        this.synthesizerReady = false;
        
        this.uiController = null;
        this.progressInterval = null;
        this.midiPlaybackContext = null;
        
        console.log('🎵 Using Fallback Audio Engine (no AudioWorklet)');
    }
    
    setUIController(uiController) {
        this.uiController = uiController;
    }
    
    async initialize() {
        try {
            // Initialize Web Audio Context
            this.audioContext = new (window.AudioContext || window.webkitAudioContext)();
            this.gainNode = this.audioContext.createGain();
            this.gainNode.connect(this.audioContext.destination);
            this.gainNode.gain.value = this.volume;
            
            this.updateStatus('Audio context initialized (fallback mode)');
            
            // Skip AudioWorklet initialization
            await this.initializeOpenMPT();
            await this.initializeSynthesizer();
            await this.loadSoundFont();
            
            // Enable audio context on user interaction
            this.setupUserActivation();
            
            this.updateStatus('Fallback audio engine ready');
            return true;
            
        } catch (error) {
            this.updateStatus('Audio engine initialization failed: ' + error.message);
            console.error('FallbackAudioEngine initialization error:', error);
            throw error;
        }
    }
    
    async initializeOpenMPT() {
        try {
            this.updateStatus('Waiting for OpenMPT/chiptune2.js...');
            
            // Wait for libraries to load
            let attempts = 0;
            while (typeof ChiptuneJsConfig === 'undefined' && typeof Module === 'undefined' && attempts < 50) {
                await new Promise(resolve => setTimeout(resolve, 100));
                attempts++;
            }
            
            if (typeof ChiptuneJsConfig !== 'undefined' && typeof ChiptuneJsPlayer !== 'undefined') {
                try {
                    this.chiptunePlayer = new ChiptuneJsPlayer(new ChiptuneJsConfig(-1, this.audioContext));
                    this.updateStatus('chiptune2.js loaded successfully');
                } catch (chipErr) {
                    console.error('ChiptuneJS initialization error:', chipErr);
                    this.updateStatus('ChiptuneJS failed to initialize, using raw OpenMPT');
                }
            } else if (typeof Module !== 'undefined') {
                this.updateStatus('libopenmpt loaded successfully');
            } else {
                this.updateStatus('OpenMPT.js failed to load - tracker modules may not play');
            }
        } catch (error) {
            this.updateStatus('OpenMPT initialization error: ' + error.message);
            console.error('OpenMPT initialization failed:', error);
        }
    }
    
    async initializeSynthesizer() {
        try {
            this.updateStatus('Loading MIDI Synthesizers...');
            
            // Try WebAudioTinySynth first (most reliable)
            if (typeof WebAudioTinySynth !== 'undefined') {
                try {
                    this.synthesizer = new WebAudioTinySynth({
                        quality: 1,
                        useReverb: 1,
                        voices: 32
                    });
                    this.synthesizerReady = true;
                    this.updateStatus('WebAudio-TinySynth loaded successfully');
                    return;
                } catch (tinyError) {
                    console.warn('WebAudioTinySynth failed:', tinyError);
                }
            }
            
            // Try JS-Synthesizer as fallback
            if (typeof JSSynth !== 'undefined') {
                try {
                    await JSSynth.waitForReady();
                    this.synthesizer = new JSSynth.Synthesizer();
                    await this.synthesizer.init(this.audioContext.sampleRate);
                    this.synthesizerReady = true;
                    this.updateStatus('JS-Synthesizer loaded successfully');
                    return;
                } catch (jsError) {
                    console.warn('JS-Synthesizer failed:', jsError);
                }
            }
            
            // Last resort: create a simple fallback synthesizer
            this.synthesizer = this.createSimpleSynthesizer();
            this.synthesizerReady = true;
            this.updateStatus('Using simple fallback synthesizer');
            
        } catch (error) {
            this.updateStatus('All MIDI synthesizers failed: ' + error.message);
            console.error('Synthesizer initialization failed:', error);
        }
    }
    
    createSimpleSynthesizer() {
        const oscillators = new Map();
        const gainNodes = new Map();
        
        return {
            send: (midiData) => {
                const [status, note, velocity] = midiData;
                const command = status & 0xF0;
                
                if (command === 0x90 && velocity > 0) {
                    this.playNote(note, velocity, oscillators, gainNodes);
                } else if (command === 0x80 || (command === 0x90 && velocity === 0)) {
                    this.stopNote(note, oscillators, gainNodes);
                }
            },
            stopMIDI: () => {
                oscillators.forEach(osc => {
                    try { osc.stop(); } catch (e) {}
                });
                oscillators.clear();
                gainNodes.clear();
            }
        };
    }
    
    playNote(note, velocity, oscillators, gainNodes) {
        try {
            this.stopNote(note, oscillators, gainNodes);
            
            const osc = this.audioContext.createOscillator();
            const gain = this.audioContext.createGain();
            
            const frequency = 440 * Math.pow(2, (note - 69) / 12);
            osc.frequency.setValueAtTime(frequency, this.audioContext.currentTime);
            osc.type = 'sawtooth';
            
            const volume = (velocity / 127) * 0.1;
            gain.gain.setValueAtTime(volume, this.audioContext.currentTime);
            
            osc.connect(gain);
            gain.connect(this.gainNode);
            osc.start();
            
            oscillators.set(note, osc);
            gainNodes.set(note, gain);
            
        } catch (error) {
            console.error('Error playing note:', error);
        }
    }
    
    stopNote(note, oscillators, gainNodes) {
        try {
            const osc = oscillators.get(note);
            const gain = gainNodes.get(note);
            
            if (osc && gain) {
                gain.gain.exponentialRampToValueAtTime(0.001, this.audioContext.currentTime + 0.1);
                osc.stop(this.audioContext.currentTime + 0.1);
                oscillators.delete(note);
                gainNodes.delete(note);
            }
        } catch (error) {
            console.error('Error stopping note:', error);
        }
    }
    
    async loadSoundFont() {
        // Skip SoundFont loading for TinySynth as it has built-in sounds
        if (this.synthesizer instanceof WebAudioTinySynth) {
            this.updateStatus('TinySynth ready with built-in sounds');
            return;
        }
        
        // For other synthesizers, try to load SoundFont
        try {
            this.updateStatus('Loading SoundFont...');
            
            const response = await fetch('/soundfonts/default.sf2');
            if (!response.ok) {
                throw new Error('SoundFont not found');
            }
            
            const soundFontData = await response.arrayBuffer();
            
            if (this.synthesizer && this.synthesizer.loadSoundFont) {
                await this.synthesizer.loadSoundFont(new Uint8Array(soundFontData));
                this.updateStatus('SoundFont loaded successfully');
            } else {
                this.updateStatus('Synthesizer ready with built-in sounds');
            }
            
        } catch (error) {
            this.updateStatus('SoundFont loading failed, using built-in sounds: ' + error.message);
        }
    }
    
    setupUserActivation() {
        const activateAudio = async () => {
            if (this.audioContext && this.audioContext.state === 'suspended') {
                await this.audioContext.resume();
                this.updateStatus('Audio context activated');
            }
        };
        
        document.addEventListener('click', activateAudio, { once: true });
        document.addEventListener('touchstart', activateAudio, { once: true });
    }
    
    async playTrack(trackData) {
        if (!trackData) {
            throw new Error('No track data provided');
        }
        
        this.stop(); // Stop any current playback
        
        const trackUrl = `/music/${trackData.filename}`;
        this.updateStatus(`Loading ${trackData.filename}...`);
        
        try {
            if (trackData.type === 'tracker') {
                await this.playTrackerModule(trackUrl);
            } else if (trackData.type === 'midi') {
                await this.playMidiFile(trackUrl, trackData);
            } else {
                throw new Error('Unsupported file type');
            }
        } catch (error) {
            this.updateStatus(`Playback failed: ${error.message}`);
            throw error;
        }
    }
    
    async playTrackerModule(url) {
        try {
            if (this.chiptunePlayer) {
                await this.playWithChiptune2(url);
            } else {
                throw new Error('No tracker player available');
            }
        } catch (error) {
            throw new Error('Failed to load tracker module: ' + error.message);
        }
    }
    
    async playWithChiptune2(url) {
        const response = await fetch(url);
        const arrayBuffer = await response.arrayBuffer();
        
        try {
            const player = this.chiptunePlayer;
            
            if (player && typeof player.load === 'function') {
                // Load the module
                await new Promise((resolve, reject) => {
                    if (player.load.length === 1) {
                        // Promise-based API
                        Promise.resolve(player.load(arrayBuffer))
                            .then(() => {
                                player.play();
                                resolve();
                            })
                            .catch(reject);
                    } else {
                        // Callback-based API
                        player.load(arrayBuffer, (error) => {
                            if (error) {
                                reject(error);
                                return;
                            }
                            player.play();
                            resolve();
                        });
                    }
                });
                
                this.isPlaying = true;
                this.updateStatus('Playing tracker module with chiptune2.js');
                this.startChiptune2Progress();
                
            } else {
                throw new Error('Chiptune2.js player not properly initialized');
            }
        } catch (error) {
            console.error('Chiptune2.js playback failed:', error);
            throw error;
        }
    }
    
    startChiptune2Progress() {
        this.progressInterval = setInterval(() => {
            if (!this.isPlaying || !this.chiptunePlayer) {
                clearInterval(this.progressInterval);
                return;
            }
            
            try {
                if (this.chiptunePlayer.currentPlayingNode) {
                    const position = this.chiptunePlayer.currentPlayingNode.getPosition();
                    const duration = this.chiptunePlayer.currentPlayingNode.getDuration();
                    
                    if (position !== undefined && duration !== undefined) {
                        this.currentTime = position;
                        this.duration = duration;
                        
                        if (this.uiController) {
                            this.uiController.updateProgress(this.currentTime, this.duration);
                        }
                        
                        // Check if playback ended
                        if (position >= duration) {
                            this.isPlaying = false;
                            if (this.uiController) {
                                this.uiController.updateControls(false, false);
                                this.uiController.nextTrack();
                            }
                            clearInterval(this.progressInterval);
                        }
                    }
                }
            } catch (error) {
                console.error('Progress tracking error:', error);
            }
        }, 100);
    }
    
    async playMidiFile(url, trackData) {
        if (!this.synthesizerReady) {
            throw new Error('MIDI synthesizer not ready');
        }
        
        try {
            const response = await fetch(url);
            const arrayBuffer = await response.arrayBuffer();
            const midiData = new Uint8Array(arrayBuffer);
            
            // Use TinySynth's built-in MIDI player if available
            if (this.synthesizer instanceof WebAudioTinySynth) {
                await this.playMidiWithTinySynth(midiData);
            } else {
                // Parse MIDI and play manually
                const midiFile = this.parseMidiFile(midiData);
                await this.playMidiEvents(midiFile);
            }
            
            this.isPlaying = true;
            this.updateStatus(`Playing MIDI: ${trackData.filename}`);
            
        } catch (error) {
            throw new Error('Failed to load MIDI file: ' + error.message);
        }
    }
    
    async playMidiWithTinySynth(midiData) {
        try {
            // Create a blob URL for the MIDI data
            const blob = new Blob([midiData], { type: 'audio/midi' });
            const url = URL.createObjectURL(blob);
            
            // Load and play with TinySynth
            if (this.synthesizer.loadMIDIUrl) {
                await this.synthesizer.loadMIDIUrl(url);
                this.synthesizer.playMIDI();
                
                this.duration = 120; // Default duration, TinySynth doesn't provide this easily
                this.startTinySynthProgress();
            } else {
                throw new Error('TinySynth MIDI loading not available');
            }
            
            // Clean up blob URL
            setTimeout(() => URL.revokeObjectURL(url), 1000);
            
        } catch (error) {
            console.error('TinySynth MIDI playback failed:', error);
            throw error;
        }
    }
    
    startTinySynthProgress() {
        let startTime = Date.now();
        
        this.progressInterval = setInterval(() => {
            if (!this.isPlaying) {
                clearInterval(this.progressInterval);
                return;
            }
            
            // Simple time tracking for TinySynth
            this.currentTime = (Date.now() - startTime) / 1000;
            
            if (this.uiController) {
                this.uiController.updateProgress(this.currentTime, this.duration);
            }
            
            // Check if playback should end (rough estimate)
            if (this.currentTime >= this.duration && this.duration > 0) {
                this.isPlaying = false;
                if (this.uiController) {
                    this.uiController.updateControls(false, false);
                    this.uiController.nextTrack();
                }
                clearInterval(this.progressInterval);
            }
        }, 100);
    }
    
    // Simple MIDI file parser
    parseMidiFile(data) {
        // Basic MIDI parsing - simplified version
        try {
            if (typeof MidiParser !== 'undefined') {
                const parser = new MidiParser();
                return parser.parse(data);
            }
        } catch (error) {
            console.warn('MIDI parser failed:', error);
        }
        
        // Return default structure if parsing fails
        return {
            events: [],
            duration: 120,
            ticksPerQuarter: 480,
            tempo: 500000
        };
    }
    
    async playMidiEvents(midiFile) {
        // Simplified MIDI event playback
        this.duration = midiFile.duration;
        
        if (midiFile.events.length === 0) {
            throw new Error('No MIDI events found');
        }
        
        // Play events in sequence
        let startTime = Date.now();
        
        this.progressInterval = setInterval(() => {
            if (!this.isPlaying) {
                clearInterval(this.progressInterval);
                return;
            }
            
            this.currentTime = (Date.now() - startTime) / 1000;
            
            if (this.uiController) {
                this.uiController.updateProgress(this.currentTime, this.duration);
            }
            
            if (this.currentTime >= this.duration) {
                this.isPlaying = false;
                if (this.uiController) {
                    this.uiController.nextTrack();
                }
                clearInterval(this.progressInterval);
            }
        }, 100);
    }
    
    pause() {
        this.isPaused = true;
        this.isPlaying = false;
        
        if (this.chiptunePlayer && this.chiptunePlayer.pause) {
            this.chiptunePlayer.pause();
        }
        
        if (this.synthesizer && this.synthesizer.stopMIDI) {
            this.synthesizer.stopMIDI();
        }
        
        if (this.progressInterval) {
            clearInterval(this.progressInterval);
        }
        
        this.updateStatus('Playback paused');
    }
    
    resume() {
        this.isPaused = false;
        this.isPlaying = true;
        
        if (this.chiptunePlayer && this.chiptunePlayer.play) {
            this.chiptunePlayer.play();
        }
        
        if (this.synthesizer && this.synthesizer.playMIDI) {
            this.synthesizer.playMIDI();
        }
        
        this.updateStatus('Playback resumed');
    }
    
    stop() {
        this.isPlaying = false;
        this.isPaused = false;
        this.currentTime = 0;
        
        // Clear progress tracking
        if (this.progressInterval) {
            clearInterval(this.progressInterval);
            this.progressInterval = null;
        }
        
        // Stop chiptune2.js playback
        if (this.chiptunePlayer && this.chiptunePlayer.stop) {
            this.chiptunePlayer.stop();
        }
        
        // Stop synthesizer
        if (this.synthesizer) {
            if (this.synthesizer.stopMIDI) {
                this.synthesizer.stopMIDI();
            }
            // Send all notes off for MIDI
            if (this.synthesizer.send) {
                for (let channel = 0; channel < 16; channel++) {
                    this.synthesizer.send([0xB0 | channel, 123, 0], 0); // All notes off
                    this.synthesizer.send([0xB0 | channel, 120, 0], 0); // All sound off
                }
            }
        }
        
        this.updateStatus('Playback stopped');
    }
    
    seek(time) {
        // Simplified seeking - restart from position
        if (this.isPlaying) {
            console.log('Seeking to', time, 'seconds (simplified implementation)');
            this.currentTime = Math.max(0, time);
        }
    }
    
    setVolume(level) {
        this.volume = Math.max(0, Math.min(1, level));
        if (this.gainNode) {
            this.gainNode.gain.value = this.volume;
        }
        
        // Set volume on synthesizer if supported
        if (this.synthesizer && this.synthesizer.setMasterVolume) {
            this.synthesizer.setMasterVolume(this.volume);
        }
    }
    
    updateStatus(message) {
        if (this.uiController) {
            this.uiController.updateSystemStatus(message);
        } else {
            console.log('FallbackAudioEngine:', message);
        }
    }
    
    // Getters for current state
    getCurrentTime() {
        return this.currentTime;
    }
    
    getDuration() {
        return this.duration;
    }
    
    getVolume() {
        return this.volume;
    }
    
    getPlayState() {
        return {
            isPlaying: this.isPlaying,
            isPaused: this.isPaused,
            currentTime: this.currentTime,
            duration: this.duration
        };
    }
}
